using System.Collections.Generic;
using System.Linq;
using Verse.Sound;
using UnityEngine;
using RimWorld;
using Verse.AI.Group;
using System.Text;
using RimWorld.Planet;

namespace Verse
{

public enum BuildingSizeCategory
{
    None,
    Small,
    Medium,
    Large
}

public class Building : ThingWithComps
{
    // Working vars - saved
    private ColorDef paintColorDef;

    // Configuration
    public bool canChangeTerrainOnDestroyed = true;

    // Constants
    private static readonly SimpleCurve ShakeAmountPerAreaCurve = new SimpleCurve()
    {
        new CurvePoint(1f, 0.07f),
        new CurvePoint(2f, 0.07f),
        new CurvePoint(4f, 0.1f),
        new CurvePoint(9f, 0.2f),
        new CurvePoint(16f, 0.5f)
    };
    
    private const float ChanceToGeneratePaintedFromTrader = 0.1f;

    private const int BuildingMinTickRate = 6;

    // Properties
    public CompPower PowerComp => GetComp<CompPower>();
    public ColorDef PaintColorDef => paintColorDef;
    public override Color DrawColor
    {
        get
        {
            if (paintColorDef != null)
                return paintColorDef.color;
            return base.DrawColor;
        }
    }
    public virtual bool TransmitsPowerNow
    {
        get
        {
            //Designed to be overridden
            //In base game this always just returns the value in the powercomp's def
            CompPower pc = PowerComp;
            return pc != null && pc.Props.transmitsPower;
        }
    }
    public override int HitPoints
    {
        set
        {
            int oldHitPoints = HitPoints;
            base.HitPoints = value;

            BuildingsDamageSectionLayerUtility.Notify_BuildingHitPointsChanged(this, oldHitPoints);
            
            if (Spawned)
                Map.events.Notify_BuildingHitPointsChanged(this);
        }
    }
    
    public virtual int MaxItemsInCell => def.building.maxItemsInCell;
    
    public bool IsClearableFreeBuilding => def.useHitPoints == false && def.scatterableOnMapGen == false && 
                                           def.passability == Traversability.Standable && this.GetStatValue(StatDefOf.WorkToBuild) == 0f;

    public virtual bool IsAirtight
    {
        get
        {
            if (def.building.isAirtight)
                return true;

            if (def.building.isStuffableAirtight && Stuff.stuffProps.isAirtight)
                return true;

            return false;
        }
    }

    public virtual bool ExchangeVacuum => !IsAirtight || def.building.alwaysExchangeVacuum;

    protected override int MinTickIntervalRate => BuildingMinTickRate;
    
    public override void ExposeData()
    {
        base.ExposeData();

        Scribe_Values.Look(ref canChangeTerrainOnDestroyed, "canChangeTerrainOnDestroyed", true);
        Scribe_Defs.Look(ref paintColorDef, "paintColorDef");
    }

    public override void SpawnSetup(Map map, bool respawningAfterLoad)
    {
        //Before base.SpawnSetup() so when regions are rebuilt this building can be accessed via edificeGrid
        if( def.IsEdifice() )
        {
            map.edificeGrid.Register(this);

            if (def.Fillage == FillCategory.Full)
                map.terrainGrid.Drawer.SetDirty();

            if (def.AffectsFertility)
                map.fertilityGrid.Drawer.SetDirty();
        }

        base.SpawnSetup(map, respawningAfterLoad);

        Map.listerBuildings.Add(this);
        
        //Remake terrain meshes with new underwall under me
        if( def.coversFloor )
            Map.mapDrawer.MapMeshDirty(Position, MapMeshFlagDefOf.Terrain, true, false);

        var occRect = this.OccupiedRect();
        for( int z=occRect.minZ; z<=occRect.maxZ; z++ )
        {
            for( int x=occRect.minX; x<=occRect.maxX; x++ )
            {
                var c = new IntVec3(x,0,z);
                Map.mapDrawer.MapMeshDirty( c, MapMeshFlagDefOf.Buildings );
                
                if (def.blockLight)
                    Map.glowGrid.LightBlockerAdded(c);
                
                if( !SnowGrid.CanCoexistWithSnow(def) )
                    Map.snowGrid.SetDepth(c, 0);
                
                if(ModsConfig.OdysseyActive && !SandGrid.CanCoexistWithSand(def))
                    Map.sandGrid.SetDepth(c, 0);
            }
        }

        if( Faction == Faction.OfPlayer )
        {
            if( def.building != null && def.building.spawnedConceptLearnOpportunity != null )
            {
                LessonAutoActivator.TeachOpportunity( def.building.spawnedConceptLearnOpportunity, OpportunityType.GoodToKnow );
            }
        }

        AutoHomeAreaMaker.Notify_BuildingSpawned( this );

        Map.listerBuildingsRepairable.Notify_BuildingSpawned(this);
        Map.listerArtificialBuildingsForMeditation.Notify_BuildingSpawned(this);
        Map.listerBuldingOfDefInProximity.Notify_BuildingSpawned(this);
        Map.listerBuildingWithTagInProximity.Notify_BuildingSpawned(this);
        
        if( !this.CanBeSeenOver() )
            Map.exitMapGrid.Notify_LOSBlockerSpawned();

        SmoothSurfaceDesignatorUtility.Notify_BuildingSpawned(this);

        // Must go after adding to buildings list
        map.events.Notify_BuildingSpawned(this);
    }

    public override void DeSpawn(DestroyMode mode = DestroyMode.Vanish)
    {
        var map = Map; // before DeSpawn!
        var attachedBuildings = GenConstruct.GetAttachedBuildings(this);

        base.DeSpawn(mode);
        
        // Find and destroy all attached buildings unless the frame is vanishing due to being completed
        if (!(def.IsFrame && mode == DestroyMode.Vanish) && !(def.building.isNaturalRock && mode == DestroyMode.WillReplace) && mode != DestroyMode.WillReplace)
        {
            foreach (var t in attachedBuildings)
            {
                if (t.def.Minifiable)
                    GenSpawn.Spawn(t.MakeMinified(), t.Position, map);
                else
                    t.Destroy(mode);
            }
        }

        if( def.IsEdifice() )
        {
            map.edificeGrid.DeRegister(this);

            if (def.Fillage == FillCategory.Full)
            {
                map.terrainGrid.Drawer.SetDirty();

                if (ModsConfig.BiotechActive)
                    map.pollutionGrid.Drawer.SetDirty();
            }

            if (def.AffectsFertility)
                map.fertilityGrid.Drawer.SetDirty();
        }

        if( mode != DestroyMode.WillReplace )
        {
            if( def.MakeFog )
                map.fogGrid.Notify_FogBlockerRemoved(this);

            if( def.holdsRoof )
                RoofCollapseCellsFinder.Notify_RoofHolderDespawned(this, map);
        
            if( def.IsSmoothable )
                SmoothSurfaceDesignatorUtility.Notify_BuildingDespawned(this, map);
        }

        CellRect occRect = this.OccupiedRect();
        for( int z=occRect.minZ; z<=occRect.maxZ; z++ )
        {
            for( int x=occRect.minX; x<=occRect.maxX; x++ )
            {
                IntVec3 c = new IntVec3(x,0,z);

                ulong changeType = MapMeshFlagDefOf.Buildings;

                if( def.coversFloor )
                    changeType |= MapMeshFlagDefOf.Terrain;

                if( def.Fillage == FillCategory.Full )
                {
                    changeType |= MapMeshFlagDefOf.Roofs;
                    changeType |= MapMeshFlagDefOf.Snow;
                    
                    if( ModsConfig.OdysseyActive )
                        changeType |= MapMeshFlagDefOf.Sand;
                }

                map.mapDrawer.MapMeshDirty( c, changeType );
                
                if (def.blockLight)
                    map.glowGrid.LightBlockerRemoved(c);
            }
        }

        map.listerBuildings.Remove(this);
        map.listerBuildingsRepairable.Notify_BuildingDeSpawned(this);
        map.listerArtificialBuildingsForMeditation.Notify_BuildingDeSpawned(this);
        map.listerBuldingOfDefInProximity.Notify_BuildingDeSpawned(this);
        map.listerBuildingWithTagInProximity.Notify_BuildingDeSpawned(this);

        if( def.building.leaveTerrain != null && Current.ProgramState == ProgramState.Playing && canChangeTerrainOnDestroyed )
        {
            foreach (var tile in this.OccupiedRect())
            {
                map.terrainGrid.SetTerrain(tile, def.building.leaveTerrain);
            }
        }

        // Mining, planning, etc
        map.designationManager.Notify_BuildingDespawned(this);

        if( !this.CanBeSeenOver() )
            map.exitMapGrid.Notify_LOSBlockerDespawned();

        if( def.building.hasFuelingPort )
        {
            var fuelingPortCell = FuelingPortUtility.GetFuelingPortCell(Position, Rotation);
            var launchable = FuelingPortUtility.LaunchableAt(fuelingPortCell, map);

            if( launchable != null && !BeingTransportedOnGravship )
                launchable.Notify_FuelingPortSourceDeSpawned();
        }

        // Must go after removing from buildings list
        map.events.Notify_BuildingDespawned(this);

        // Scatter items if there were multiple items per cell
        if( MaxItemsInCell >= 2 )
        {
            foreach( var c in this.OccupiedRect() )
            {
                int items = c.GetItemCount(map);
                if( items <= 1 || items <= c.GetMaxItemsAllowedInCell(map) )
                    continue;

                for( int i = 0; i < items - 1; i++ )
                {
                    var firstItem = c.GetFirstItem(map);
                    if( firstItem != null )
                    {
                        firstItem.DeSpawn();
                        GenPlace.TryPlaceThing(firstItem, c, map, ThingPlaceMode.Near);

                        if( c.GetItemCount(map) <= c.GetMaxItemsAllowedInCell(map) )
                            break;
                    }
                    else
                        break;
                }
            }
        }
    }
    
    public override void Destroy(DestroyMode mode = DestroyMode.Vanish)
    {
        bool wasSpawned = Spawned;
        var map = Map; // before Destroy()!
        
        // before Destroy(); the math is easier to do here
        SmoothableWallUtility.Notify_BuildingDestroying(this, mode);

        var lord = this.GetLord();
        if( lord != null )
            lord.Notify_BuildingLost(this);

        base.Destroy(mode);

        // (buildings can be reinstalled)
        InstallBlueprintUtility.CancelBlueprintsFor(this);

        if (wasSpawned)
        {
            if (mode == DestroyMode.Deconstruct)
                SoundDefOf.Building_Deconstructed.PlayOneShot(new TargetInfo(Position, map));
            else if (mode == DestroyMode.KillFinalize)
                DoDestroyEffects(map);
        }

        if (wasSpawned)
        {
            var blueprint = ThingUtility.CheckAutoRebuildOnDestroyed(this, mode, map, def);
            if (blueprint is Blueprint_Storage storageBP && this is Building_Storage storage)
            {
                storageBP.SetStorageGroup(storage.storageGroup);
                storageBP.settings = new StorageSettings();
                storageBP.settings.CopyFrom(storage.settings);
            }
        }
    }

    public override void SetFaction( Faction newFaction, Pawn recruiter = null )
    {
        if( Spawned )
        {
            Map.listerBuildingsRepairable.Notify_BuildingDeSpawned(this);
            Map.listerBuildingWithTagInProximity.Notify_BuildingDeSpawned(this);
            Map.listerBuildings.Remove(this);
        }

        base.SetFaction(newFaction, recruiter);

        if( Spawned )
        {
            Map.listerBuildingsRepairable.Notify_BuildingSpawned(this);
            Map.listerArtificialBuildingsForMeditation.Notify_BuildingSpawned(this);
            Map.listerBuildingWithTagInProximity.Notify_BuildingSpawned(this);
            Map.listerBuildings.Add(this);
            Map.mapDrawer.MapMeshDirty(Position, MapMeshFlagDefOf.PowerGrid, true, false);

            if( newFaction == Faction.OfPlayer )
                AutoHomeAreaMaker.Notify_BuildingClaimed(this);
        }
    }

    public override void PreApplyDamage(ref DamageInfo dinfo, out bool absorbed)
    {
        if (def.building != null && !def.building.canBeDamagedByAttacks)
        {
            absorbed = true;
            return;
        }
        
        if( Faction != null && Spawned && Faction != Faction.OfPlayer )
        {
            for( int i=0; i<Map.lordManager.lords.Count; i++ )
            {
                var lord = Map.lordManager.lords[i];
                if( lord.faction == Faction )
                    lord.Notify_BuildingDamaged(this, dinfo);
            }
        }

        base.PreApplyDamage(ref dinfo, out absorbed);

        if (!absorbed && Faction != null)
            Faction.Notify_BuildingTookDamage(this, dinfo);
        
        if(!absorbed)
            GetComp<CompStunnable>()?.ApplyDamage(dinfo);
    }

    public override void PostApplyDamage(DamageInfo dinfo, float totalDamageDealt)
    {
        base.PostApplyDamage(dinfo, totalDamageDealt);

        if( Spawned )
            Map.listerBuildingsRepairable.Notify_BuildingTookDamage(this);
    }

    public override void DrawExtraSelectionOverlays()
    {
        base.DrawExtraSelectionOverlays();

        var ebp = InstallBlueprintUtility.ExistingBlueprintFor(this);

        if( ebp != null )
            GenDraw.DrawLineBetween(this.TrueCenter(), ebp.TrueCenter());
    }

    public override IEnumerable<Gizmo> GetGizmos()
    {
        foreach( var c in base.GetGizmos() )
        {
            yield return c;
        }

        if( ((def.BuildableByPlayer && def.passability != Traversability.Impassable && !def.IsDoor) || def.building.forceShowRoomStats)
            && Gizmo_RoomStats.GetRoomToShowStatsFor(this) != null
            && Find.Selector.SingleSelectedObject == this )
        {
            yield return new Gizmo_RoomStats(this);
        }
        
        var selectMonumentMarker = QuestUtility.GetSelectMonumentMarkerGizmo(this);
        if( selectMonumentMarker != null )
            yield return selectMonumentMarker;

        if( def.Minifiable && (Faction == Faction.OfPlayer || def.building.alwaysUninstallable) )
            yield return InstallationDesignatorDatabase.DesignatorFor(def);

        ColorInt? glowColorOverride = null;
        if (GetComp<CompGlower>() is CompGlower glower && glower.HasGlowColorOverride)
            glowColorOverride = glower.GlowColor;

        if (!def.building.neverBuildable)
        {
            var buildCopy = BuildCopyCommandUtility.BuildCopyCommand(def, Stuff, StyleSourcePrecept as Precept_Building, StyleDef, true, glowColorOverride);
            if (buildCopy != null)
                yield return buildCopy;
        }

        if( Faction == Faction.OfPlayer || def.building.alwaysShowRelatedBuildCommands )
        {
            foreach( var facility in BuildRelatedCommandUtility.RelatedBuildCommands(def) )
            {
                yield return facility;
            }
        }

        if (this.GetLord() is Lord lord && lord.CurLordToil is LordToil)
        {
            foreach (var gizmo in lord.CurLordToil.GetBuildingGizmos(this))
            {
                yield return gizmo;
            }
        }
    }

    public override IEnumerable<StatDrawEntry> SpecialDisplayStats()
    {
        foreach (var s in base.SpecialDisplayStats())
        {
            yield return s;
        }

        if (PaintColorDef != null && !PaintColorDef.label.NullOrEmpty())
            yield return new StatDrawEntry(StatCategoryDefOf.Building, "Stat_Building_PaintColor".Translate(), PaintColorDef.LabelCap, "Stat_Building_PaintColorDesc".Translate(), StatDisplayOrder.Thing_Paintable);
    }

    public override AcceptanceReport ClaimableBy(Faction by)
    {
        if( !def.Claimable )
            return false;

        // Note if Faction is null we don't want to check Map.ParentFaction because we want players to be able to claim
        // abandoned structures on the their own map.
        if (Faction == by)
            return false;

        for ( int i = 0; i < AllComps.Count; i++ )
        {
            if( AllComps[i].CompPreventClaimingBy(by) )
                return false;
        }

        if (FactionPreventsClaimingOrAdopting(Faction ?? Map?.ParentFaction, true, out var reason))
            return reason;

        return true;
    }

    public virtual AcceptanceReport DeconstructibleBy(Faction faction)
    {
        for (var i = 0; i < AllComps.Count; i++)
        {
            if (AllComps[i].CompForceDeconstructable())
                return true;
        }

        if( !def.building.IsDeconstructible )
            return false;

        if( DebugSettings.godMode )
            return true;

        if (Faction == faction)
            return true;

        if (def.building.alwaysDeconstructible)
            return true;

        return ClaimableBy(faction);
    }

    public virtual ushort PathWalkCostFor(Pawn p)
    {
        return 0;
    }

    public virtual bool IsDangerousFor(Pawn p)
    {
        return false;
    }

    public virtual bool IsWorking() => true;
    private void DoDestroyEffects(Map map)
    {
        if( def.building.destroyEffecter != null && !Position.Fogged(map) )
        {
            var effecter = def.building.destroyEffecter.Spawn(Position, map);
            effecter.Trigger(new TargetInfo(Position, map), TargetInfo.Invalid);
            effecter.Cleanup();
            return; // if there is a custom effecter, let it handle everything
        }

        if (!def.IsEdifice())
            return;

        GetDestroySound()?.PlayOneShot(new TargetInfo(Position, map));

        foreach (var c in this.OccupiedRect())
        {
            int dustAmount = def.building.isNaturalRock ? 1 : Rand.RangeInclusive(3, 5);
            for (int i = 0; i < dustAmount; i++)
            {
                FleckMaker.ThrowDustPuffThick(c.ToVector3Shifted(), map, Rand.Range(1.5f, 2f), Color.white);
            }
        }

        if (Find.CurrentMap == map)
        {
            float shake = def.building.destroyShakeAmount;

            if (shake < 0f)
                shake = ShakeAmountPerAreaCurve.Evaluate(def.Size.Area);

            //Don't shake for expired buildings
            var lifespan = this.TryGetComp<CompLifespan>();
            if (lifespan == null || lifespan.age < lifespan.Props.lifespanTicks)
                Find.CameraDriver.shaker.DoShake(shake);
        }
    }

    private SoundDef GetDestroySound()
    {
        const int SizeSmall = 1;
        const int SizeMedium = 4;

        //Explicitly defined
        if (!def.building.destroySound.NullOrUndefined())
            return def.building.destroySound;

        StuffCategoryDef stuffCategory;
        if (def.MadeFromStuff && Stuff != null && !Stuff.stuffProps.categories.NullOrEmpty())
            stuffCategory = Stuff.stuffProps.categories[0];
        else if (!def.CostList.NullOrEmpty() && def.CostList[0].thingDef.IsStuff && !def.CostList[0].thingDef.stuffProps.categories.NullOrEmpty())
            stuffCategory = def.CostList[0].thingDef.stuffProps.categories[0];
        else
            return null;

        switch (def.building.buildingSizeCategory)
        {
            case BuildingSizeCategory.Small:
                if (!stuffCategory.destroySoundSmall.NullOrUndefined())
                    return stuffCategory.destroySoundSmall;
                break;
            case BuildingSizeCategory.Medium:
                if (!stuffCategory.destroySoundMedium.NullOrUndefined())
                    return stuffCategory.destroySoundMedium;
                break;
            case BuildingSizeCategory.Large:
                if (!stuffCategory.destroySoundLarge.NullOrUndefined())
                    return stuffCategory.destroySoundLarge;
                break;
            case BuildingSizeCategory.None: //Based on actual building size
                var area = def.Size.Area;

                if (area <= SizeSmall && !stuffCategory.destroySoundSmall.NullOrUndefined())
                    return stuffCategory.destroySoundSmall;
                else if (area <= SizeMedium && !stuffCategory.destroySoundMedium.NullOrUndefined())
                    return stuffCategory.destroySoundMedium;
                else if (!stuffCategory.destroySoundLarge.NullOrUndefined())
                    return stuffCategory.destroySoundLarge;
                break;
        }

        return null;
    }

    public override void PostGeneratedForTrader(TraderKindDef trader, PlanetTile forTile, Faction forFaction)
    {
        base.PostGeneratedForTrader(trader, forTile, forFaction);

        if (def.building.paintable && Rand.Value < ChanceToGeneratePaintedFromTrader)
            ChangePaint(DefDatabase<ColorDef>.AllDefs.Where(x => x.colorType == ColorType.Structure).RandomElement());
        else if (def.colorGeneratorInTraderStock != null)
            this.SetColor(def.colorGeneratorInTraderStock.NewRandomizedColor(), reportFailure: true);
    }

    public override string GetInspectStringLowPriority()
    {
        var str = base.GetInspectStringLowPriority();

        if (!DeconstructibleBy(Faction.OfPlayer) && (def.IsNonDeconstructibleAttackableBuilding || def.building.quickTargetable) && def.building.displayAttackToDestroyOnInspectPane)
        {
            if(!str.NullOrEmpty())
                str += "\n";

            str += "AttackToDestroy".Translate(); 
        }

        return str;
    }

    public void ChangePaint(ColorDef colorDef)
    {
        paintColorDef = colorDef;
        Notify_ColorChanged();

        if (Spawned && def.drawerType is DrawerType.MapMeshOnly or DrawerType.MapMeshAndRealTime)
            Map.mapDrawer.MapMeshDirty(Position, MapMeshFlagDefOf.Buildings);
    }

    public static Gizmo SelectContainedItemGizmo(Thing container, Thing item)
    {
        if (!container.Faction.IsPlayerSafe())
            return null;

        return ContainingSelectionUtility.SelectCarriedThingGizmo(container, item);
    }
    
    public virtual int HaulToContainerDuration(Thing thing)
    {
        return def.building.haulToContainerDuration;
    } 
}

}
